//
//  TodoList.swift
//  Do It
//
//  Created by Jim Dovey on 8/25/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI

struct TodoList: View {
    private enum ListData {
        case list(TodoItemListStruct)
        case items(LocalizedStringKey, [TodoItemStruct])
        case group(TodoItemGroup)
    }
    
    private enum Sheet: Identifiable, Hashable {
        case itemEditor
        case listEditor
        
        var id: Sheet { self }
    }
    
    @State private var presentedSheet: Sheet? = nil

    @State private var sortBy: SortOption = .manual
    @State private var showingChooser: Bool = false
    @Environment(\.presentationMode) private var presentationMode
    @EnvironmentObject private var data: DataCenter

    private static let itemTemplate = TodoItemStruct(
        id: Int.min, title: "New Item", priority: .normal,
        notes: nil, date: nil, listID: 2002, complete: false)

    @State private var editingItem = Self.itemTemplate

    @State private var listData: ListData

    init(list: TodoItemListStruct) {
        self._listData = State(wrappedValue: .list(list))
    }

    init(title: LocalizedStringKey, items: [TodoItemStruct]) {
        self._listData = State(wrappedValue: .items(title, items))
    }

    init(group: TodoItemGroup) {
        self._listData = State(wrappedValue: .group(group))
    }

    var body: some View {
        List {
            ForEach(sortedItems) { item in
                NavigationLink(destination: TodoItemDetail(item: item)
                    .environmentObject(self.data)
                ) {
                    TodoItemRow(item: item)
                        .accentColor(self.color(for: item))
                }
            }
            .onDelete { self.removeTodoItems(atOffsets: $0) }
            .onMove(perform: self.sortBy == .manual
                ? { self.moveTodoItems(fromOffsets: $0, to: $1) }
                : nil)
        }
        .navigationBarTitle(title)
        .navigationBarItems(trailing: barItems)
        .listStyle(GroupedListStyle())
        .actionSheet(isPresented: $showingChooser) {
            ActionSheet(
                title: Text("Sort Order"),
                buttons: SortOption.allCases.map { opt in
                    ActionSheet.Button.default(Text(opt.title)) {
                        self.sortBy = opt
                    }
            })
        }
        .sheet(item: $presentedSheet, content: presentEditor(of:))
        .onReceive(data.$todoItems.combineLatest(data.$todoLists)) { _ in
            self.updateData()
        }
    }
}

// MARK: - Helper Properties

extension TodoList {
    private var sortButton: some View {
        Button(action: { self.showingChooser.toggle() }) {
            Image(systemName: "arrow.up.arrow.down.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Sort List"))
        }
    }

    private var addButton: some View {
        Button(action: {
            self.editingItem = Self.itemTemplate
            self.presentedSheet = .itemEditor
        }) {
            Image(systemName: "plus.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Add New To-Do Item"))
        }
    }

    private var editorSheet: some View {
        let done = Button(action:{
            self.data.addTodoItem(self.editingItem)
            self.presentedSheet = nil
        }) {
            Text("Done")
                .bold()
        }
        let cancel = Button("Cancel") {
            self.presentedSheet = nil
        }
        return NavigationView {
            TodoItemEditor(item: $editingItem)
                .navigationBarItems(leading: cancel, trailing: done)
                .environmentObject(data)
        }
    }

    private var barItems: some View {
        HStack(spacing: 14) {
            if isList {
                Button(action: { self.presentedSheet = .listEditor }) {
                    Image(systemName: "info.circle.fill")
                        .imageScale(.large)
                        .accessibility(label: Text("Show List Information"))
                }
            }
            sortButton
            addButton
            EditButton()
        }
    }
    
    private func presentEditor(of type: Sheet) -> some View {
        switch type {
        case .itemEditor:
            return AnyView(self.editorSheet)
        case .listEditor:
            return AnyView(TodoListEditor(list: self.list!)
                .environmentObject(self.data))
        }
    }

    private var isList: Bool {
        if case .list = self.listData {
            return true
        }
        return false
    }
    
    private var list: TodoItemListStruct? {
        if case .list(let list) = self.listData {
            return list
        }
        return nil
    }

    private func forciblyDismiss() {
        presentationMode.wrappedValue.dismiss()
    }

    private var items: [TodoItemStruct] {
        switch listData {
        case .list(let list): return data.items(in: list)
        case .items(_, let items): return items
        case .group(let group): return group.items(from: data)
        }
    }

    private var title: LocalizedStringKey {
        switch listData {
        case .list(let list): return LocalizedStringKey(list.name)
        case .items(let name, _): return name
        case .group(let group): return group.title
        }
    }

    private func color(for item: TodoItemStruct) -> Color {
        switch listData {
        case .list(let list): return list.color.uiColor
        case .items: return data.list(for: item).color.uiColor
        case .group(let group): return group.color
        }
    }
}

// MARK: - Sorting

extension TodoList {
    private var sortedItems: [TodoItemStruct] {
        if case .manual = sortBy { return items }

        return items.sorted {
            switch sortBy {
            case .title:
                return $0.title.lowercased() < $1.title.lowercased()
            case .priority:
                return $0.priority > $1.priority
            case .dueDate:
                return ($0.date ?? .distantFuture) <
                    ($1.date ?? .distantFuture)
            case .manual:
                fatalError("unreachable")
            }
        }
    }
}

// MARK: - Model Manipulation

extension TodoList {
    private var usingUnchangedList: Bool {
        sortBy == .manual
    }

    private func translate(offsets: IndexSet) -> IndexSet {
        guard !usingUnchangedList else { return offsets }
        let items = sortedItems // calculate this just once
        return IndexSet(offsets.map { index in
            data.todoItems.firstIndex { $0.id == items[index].id }!
        })
    }

    private func translate(index: Int) -> Int {
        guard !usingUnchangedList else { return index }
        return data.todoItems.firstIndex { $0.id == sortedItems[index].id }!
    }

    private func removeTodoItems(atOffsets offsets: IndexSet) {
        let realOffsets = translate(offsets: offsets)
        data.removeTodoItems(atOffsets: realOffsets)
    }

    private func moveTodoItems(fromOffsets offsets: IndexSet, to newIndex: Int) {
        let realOffsets = translate(offsets: offsets)
        let realIndex = translate(index: newIndex)
        data.moveTodoItems(fromOffsets: realOffsets, to: realIndex)
    }

    private func updateData() {
        switch listData {
        case let .items(title, items):
            let newItems = data.items(withIDs: items.map { $0.id })
            listData = .items(title, newItems)

        case let .list(list):
            if let newList = data.todoLists.first(where: { $0.id == list.id }) {
                listData = .list(newList)
            }
            else {
                // List is gone!
                forciblyDismiss()
            }

        case .group:
            break
        }
    }
}

fileprivate enum SortOption: String, CaseIterable {
    case title = "Title"
    case priority = "Priority"
    case dueDate = "Due Date"
    case manual = "Manual"

    var title: LocalizedStringKey { LocalizedStringKey(rawValue) }
}

struct TodoList_Previews: PreviewProvider {
    static var previews: some View {
        let data = DataCenter()
        return NavigationView {
            TodoList(title: "All Items", items: data.todoItems)
                .environmentObject(DataCenter())
        }
    }
}
